package kom.itbeankit.common;

import kom.itbeankit.common.DataPool;
import kom.itbeankit.common.ItBeanPin;
import java.awt.*;
import java.util.*;
import kom.itbeankit.common.ItBeanCommunicationListener;

/*******************************************************************************************
 * This abstract class implements the methods defined in StandardBean interface.
 * The methods that must be universal for all ItBeans, e.g. run() or start() of
 * Runnable interface are implemented, too.
 *******************************************************************************************
 * <code>
 * Table Of Modifications
 *
 * 02/04/98    Christian C. Hoffmann    Design
 * 08/04/98    Kerim Mansour             fireCommunicationEvent(): private->protected
 * 12/04/98    Christian C. Hoffmann    main() added
 *                                      call of main() in run() added
 * 05/05/98    Christian C. Hoffmann    main() renamed to mainloop()
 * 12/05/98    Christian C. Hoffmann	   following properties plus getter and setter added:
 *                                      first, last, FreeRunning, Pool
 * 20/05/98    Christian C. Hoffmann    property iterationNumber + set() and get() added
 * 21/05/98    Christian C. Hoffmann    item BEAN_STARTED in fireCommunicationEvent() added
 * 22/05/98    Christian C. Hoffmann    property isConnected + set() added
 * 22/06/98    Kerim Mansour            Redesign of StandardBean and ItBean, set,getVisiblitity removed, better method names
 *                                      Design as Bean compatible to Symantec VusualCafe,
										methods accepts(int int), accepts(int) added
 * </code>
 *******************************************************************************************
 * @author Christian C. Hoffmann / Kerim Mansour
 * @version 1.21 13/07/98
 ***************************************************************************************/


public abstract class ItBean implements StandardBean, java.lang.Runnable
{

	public ItBean()
	{

	}

    /**
    * This method initializes all beans fields and the display
    * variables.
    *
    */
    public abstract void initializeBean();

    /**
     * This method must be implemented by every ItBean.
     * It will contain the main thread loop waiting for
     * realstart() or any user actions.
     */
    protected abstract void mainloop();

    /**
     * This method invokes all methods to close the Frame and
     * to avoid visualising all subsequent calculations.
     * @see makeVisible()
     */
    public abstract void makeInvisible();

    /**
     * This method invokes the methods open a Frame and
     * to visualize all subsequent calculations.
     * @see makeInvisible()
     */
    public abstract void makeVisible();

 
     /**
     * This method should start a new Iteration.
     * The Number of the Iteration will be set by the construction kit or (in freeRunning) by the bean itself
     */
    public abstract void startIteration();

    /** This method checks wheather a certain Datatype can be
     * accepted by an ItBean (regardless of the Pin). (Comes
     * in handy if there is only one pin). It must be implemented by everyone.
     * @see accepts (int datatype, int pin)
     */
    public boolean accepts(int datatype)
    {
       return true;
    }

   /** This method is used to determine wheather the datatypes of two
     * ItBeans (that should be connected) are compatible.It must be implemented by everyone.
     * @param datatype The Datatype of the SourceItBean
     * @param pin The Pin the SourceItBean should be connected to
     */
    public boolean accepts(int datatype, int pin)
    {
        return true;
    }
    /* Used to get the DataPool for the in/outputfiles
    * @return DataPool
    */
	public DataPool getPool()
	{
		return this.pool;
	}
    
    /* Used to see if the bean if the bean has already been connected to others
    * via ItFilenames on the pins.
	* @return connected
	*/
	public boolean isConnected()
	{
		return this.connected;
	}

    /**
    * This method returns the actual iteration number.
    * @return int
    */
	public int getIterationNumber()
	{
		return this.iterationNumber;
	}

	/**
     * This method fires a communication event by invoking the in
     * ItBeanCommunicationListener declared methods of
     * the registrated listeners.
    * @param e common.ItBeanCommunicationEvent
     */
    protected void fireCommunicationEvent(ItBeanCommunicationEvent e) {
	if (this.communicationListener != null) {
		switch (e.getId()) {
			case e.BEAN_INTERRUPTED: this.communicationListener.beanInterrupted(e); break;
			case e.BEAN_TERMINATED: this.communicationListener.beanTerminated(e); break;
			case e.ITERATION_COMPLETED: this.communicationListener.beanCompleted(e); break;
			case e.BEAN_STARTED: this.communicationListener.beanStarted(e); break;
		}
	}
    }
 
    /**
     * This method retruns the last property which signalizes that
     * the itBean is the last itbean of a structure. The bean will
     * put its output data to a file or a network connection.
     * @return the state
     * @deprecated Replaced by isLast()
     */
    public boolean last()
	{
		return this.last;
	}

    /**
     * This method retruns the last property which signalizes that
     * the itBean is the last itbean of a structure. The bean will
     * put its output data to a file or a network connection.
     * @return the state
     */
    public boolean isLast()
	{
		return this.last;
	}

	/**
     * This method returns the First property which signalizes that
     * the itBean is the first itbean of a structure. The bean will
     * get its input data from a file or a network connection.
     * @return the first state
     * @deprecated Replaced by isFirst()
     */
	public boolean first()
	{
		return this.first;
	}

    /**
     * This method returns the First property which signalizes that
     * the itBean is the first itbean of a structure. The bean will
     * get its input data from a file or a network connection.
     * @return the first state
     */
	public boolean isFirst()
	{
		return this.first;
	}

     /**
     * This method returns the freeRunning property that notifys wether
     * the ItBean is running step by step (restartet by startItertion())
     * or uncontrolled, running all iterations.
     * @return the freeRunning state
     */
	public boolean isFreeRunning()
	{
		return this.freeRunning;
	}

    /**
     * This method returns the pins property.
     * @return common.ItBeanPin[] a twenty-element-array of ItBeanPin
     */
	public ItBeanPin[] getPins()
	{
		return this.pins;
	}

    
    /**
    * getUserLevelComplexity returns the complexity property.
    */
	public int getUserLevelComplexity()
	{
		return this.userLevelComplexity;
	}

    
    /**
    * This method returns the BeanNumber.
    * @return the bean number
    */
    public int getBeanNumber()
	{
		return this.beanNumber;
	}
	
    /**
     * This method sets the Pins property.
     * @param pins common.ItBeanPin[] a twenty-element-array of ItBeanPin
     */
	public void setPins(ItBeanPin[] pins)
	{
		this.pins = pins;
	}


    
    
    
    
    // below are methods only used by the construction kit or methods not to be changed by
    // the ItBean implementation
    
    
    
    
    
    
    
    
    /**
     * setComplexity sets the complexity property.
     */
	public void setUserLevelComplexity(int userLevelComplexity)
	{
			this.userLevelComplexity = userLevelComplexity;
	}

    /**
    * This method sets the ItBeanCommunicationListener (normally the framework)
    * @exception java.util.TooManyListenersException thrown if listener is already set
    * @param l the Listener to add
    */
	public void addItBeanCommunicationListener(ItBeanCommunicationListener l) throws java.util.TooManyListenersException
	{
		if (l != null) {
			if (this.communicationListener == null)	this.communicationListener = l;
			else throw new java.util.TooManyListenersException();
		}
	}

    /**
    * This method removes the ItBeanCommunicationListener (normally the framework)
    * @param l the Listener to remove
    */
	public void removeItBeanCommunicationListener(ItBeanCommunicationListener l)
	{
		if (l == this.communicationListener) this.communicationListener = null;
	}


   /**
     * This method sets the freeRunning property that notifys wether
     * the ItBean is running step by step (restartet by startItertion())
     * or uncontrolled, running all iterations.
     * @param flag boolean
     */
	public void setFreeRunning(boolean freeRunning)
	{
			this.freeRunning = freeRunning;
	}

    /**
     * This method sets the First property which signalizes that
     * the itBean is the first itbean of a structure. The bean will
     * get its input data from a file or a network connection.
     * @param flag
     */
	public void setFirst(boolean first)
	{
			this.first = first;
	}

    


    /**
     * This method sets the Last property which signalizes that
     * the itBean is the last itbean of a structure. The bean will
     * put its output data to a file or a network connection.
     * @param flag the new state
     */
	public void setLast(boolean Last)
	{
			this.last = Last;
	}

    
    /**
     * This method sets the state of the Connected flag. By setting
     * to true the framework notifys the bean that it is connected to
     * other beans (means its input pins filenames have been set)
     * @param state true if the bean is connected
     */
	public void setConnected(boolean connected)
	{
			this.connected = connected;
	}


    /**
     * This method sets the bean number.
     * @param number the new bean number
     */
	public void setBeanNumber(int beanNumber)
	{
			this.beanNumber = beanNumber;
	}


    /**
     * This method sets the datapool, that stores all produced data.
     * @param pool the datapool
     */
	public void setPool(DataPool pool)
	{
		this.pool = pool;
	}

 
    /**
     * This method sets the iteration number.
     * @param newIteration the new IterationNumber
     */
	public void setIterationNumber(int iterationNumber)
	{
			this.iterationNumber = iterationNumber;
	}

 
    /**
    * This method returns the classes thread.
    */
    public Thread getRunner() {
	    if (this.runner != null) {
		    return this.runner;
	    }
	    return null;
    }

    /**
     * run method initializes the bean and runs mainloop() waiting
     * for startIteration(). It will be invoked automatically
     * after you started the thread with start().
     */
    public void run() {
    	initializeBean();
    	mainloop();
    }
    

    /**
     * This method starts a separate thread.
     * @see run() for the executable bean code
     */
    public void start() {
    	if (this.runner == null); {
    		this.runner = new Thread(this);
    		this.runner.start();
        	}
   }

    /**
     * This method stops the classes thread.
     */
    public void stop() {
    	if (this.runner != null) {
    		this.runner.stop();
    		this.runner = null;
    	}
    }



	protected DataPool pool;
	protected int userLevelComplexity;
	protected int beanNumber;
	protected boolean first;
	protected ItBeanPin[] pins;
	protected boolean connected;
	protected ItBeanCommunicationListener communicationListener;
	protected boolean last;
	protected boolean freeRunning;
	protected int iterationNumber;
	protected Thread runner;
	protected boolean visible=false;

}